/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: AutoCompletePhaseListener.java,v 1.5 2006/02/04 00:43:22 inder Exp $
 */

package com.sun.j2ee.blueprints.ui.autocomplete;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.net.URLConnection;
import java.util.Iterator;
import java.util.List;
import javax.faces.FacesException;
import javax.faces.context.FacesContext;
import javax.faces.el.EvaluationException;
import javax.faces.el.MethodBinding;
import javax.faces.event.PhaseEvent;
import javax.faces.event.PhaseId;
import javax.faces.event.PhaseListener;
import javax.faces.webapp.UIComponentTag;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


/**
 * <p>
 * Phase listener which handles two types of requests:
 *  <ol>
 *  <li> Responds to requests for the JavaScript file referenced by the rendered textfield component markup</li>
 *  <li> Responds to autocompletion requests</li>
 *  </ol>
 *
 * @author Tor Norbye
 * @author Ed Burns
 */
public class AutoCompletePhaseListener implements PhaseListener {
    /** Max number of results returned in a single completion request. */
    static final int MAX_RESULTS_RETURNED = 10;
    private static final String AJAX_VIEW_ID = "auto-complete-pl.faces";

    public AutoCompletePhaseListener() {
    }

    public void afterPhase(PhaseEvent event) {
        try {
            String rootId = event.getFacesContext().getViewRoot().getViewId();
            if (rootId.indexOf(AJAX_VIEW_ID) != -1) {
                handleAjaxRequest(event);
            }
        } catch (Exception e) {
            System.out.println("AutoCompletePhaseListener afterPhase() threw an exception!");
            e.printStackTrace(System.out);
        }
    }

    /** 
     * The URL is identified as an "ajax" request, e.g. an asynchronous request,
     * so we need to extract the arguments from the request, invoke the completion
     * method, and return the results in the form of an XML response that the
     * browser JavaScript can handle.
     */
    private void handleAjaxRequest(PhaseEvent event) {
        FacesContext context = event.getFacesContext();
        HttpServletResponse response =
            (HttpServletResponse)context.getExternalContext().getResponse();

        Object object = context.getExternalContext().getRequest();

        if (!(object instanceof HttpServletRequest)) {
            // PortletRequest? Handle that here?
            return;
        }

        HttpServletRequest request = (HttpServletRequest)object;
        String prefix = request.getParameter("prefix");
        String method = request.getParameter("method");
        StringBuffer sb = new StringBuffer();
        boolean namesAdded = false;

        try {
            CompletionResult results = getCompletionItems(context, method, prefix);
            List items = results.getItems();

            // Chop off results at a max -- in case client methods
            // do the wrong thing and generate tons of data. I ought
            // to make this configurable on the component...
            int n = Math.min(MAX_RESULTS_RETURNED, items.size());

            if (n > 0) {
                sb.append("<items>");

                Iterator it = items.iterator();

                while (it.hasNext()) {
                    sb.append("<item>");
                    sb.append(it.next().toString());
                    sb.append("</item>");
                }

                sb.append("</items>");

                response.setContentType("text/xml");
                response.setHeader("Cache-Control", "no-cache");
                response.getWriter().write(sb.toString());
            } else {
                //nothing to show
                response.setStatus(HttpServletResponse.SC_NO_CONTENT);
            }

            event.getFacesContext().responseComplete();

            return;
        } catch (EvaluationException ee) {
            // log(ee.toString());
            ee.printStackTrace();
        } catch (IOException ioe) {
            // log(ioe.toString());
            ioe.printStackTrace();
        }
    }

    private CompletionResult getCompletionItems(FacesContext context, String methodExpr,
        String prefix) {
        // Find the user/component-specified completion method and invoke it. That
        // method should populate the CompletionResult object we'return passing in to it.
        if (UIComponentTag.isValueReference(methodExpr)) {
            Class[] argTypes = { FacesContext.class, String.class, CompletionResult.class };
            MethodBinding vb = context.getApplication().createMethodBinding(methodExpr, argTypes);
            CompletionResult result = new CompletionResult();
            Object[] args = { context, prefix, result };

            vb.invoke(context, args);

            return result;
        } else {
            // FIXME - localize this
            throw new FacesException("Method binding expression '" + methodExpr +
                    "' is not an expression");
        }
    }

    public void beforePhase(PhaseEvent event) {
    }

    public PhaseId getPhaseId() {
        return PhaseId.RESTORE_VIEW;
    }
}
