/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: AutoCompleteRenderer.java,v 1.11 2006/04/26 04:03:01 craig_mcc Exp $
 */

package com.sun.j2ee.blueprints.ui.autocomplete;

import com.sun.j2ee.blueprints.ui.util.Util;
import java.beans.Beans;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.faces.application.ViewHandler;
import javax.faces.component.UIComponent;
import javax.faces.component.UIViewRoot;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.Renderer;
import org.apache.shale.remoting.Mechanism;
import org.apache.shale.remoting.XhtmlHelper;

/**
 * <p>This renderer generates HTML (including JavaScript) for AutoCompleteComponents,
 * emitting the necessary markup to provide auto completion for the textfield.</p>
 *
 * @author Tor Norbye
 */
public class AutoCompleteRenderer extends Renderer {


    // ------------------------------------------------------ Manifest Constants


    /**
     * <p>The static CSS resource for this component.</p>
     */
    private static final String CSS_RESOURCE =
            "/META-INF/autocomplete/styles.css";


    /**
     * <p>The static script resource for this component.</p>
     */
    private static final String SCRIPT_RESOURCE =
            "/META-INF/autocomplete/script.js";


    /**
     * <p>Resource identifier for our completion results event handler.</p>
     */
    private static final String COMPLETION_HANDLER =
            "/bpui_autocomplete_handler/completions";


    // -------------------------------------------------------- Static Variables


    /**
     * <p>HTML attributes with boolean values that are passed through
     * to the rendered output.</p>
     */
    private static String booleanPassthruAttributes[] =
    { "disabled", "readonly", "ismap" };


    /**
     * <p>HTML attributes with integer values that are passed through
     * to the rendered output.</p>
     */
    private static String integerPassthruAttributes[] =
    { "maxlength", "size" };


    /**
     * <p>HTML attributes with String values that are passed through
     * to the rendered output.</p>
     */
    private static String passthruAttributes[] =
    { "accept", "accesskey", "alt", "bgcolor", "border",
      "cellpadding", "cellspacing", "charset", "cols",
      "coords", "dir", "enctype", "frame", "height",
      "hreflang", "lang", "longdesc", "onblur",
      "onchange", "onclick", "ondblclick", "onfocus",
      "onkeydown", "onkeypress", "onkeyup", "onload",
      "onmousedown", "onmousemove", "onmouseout",
      "onmouseover", "onmouseup", "onreset", "onselect",
      "onsubmit", "onunload", "rel", "rev", "rows",
      "rules", "shape", "style", "summary",
      "tabindex", "target", "title", "usemap", "width" };


    /**
     * <p>Stateless helper bean to manufacture resource linkages.</p>
     */
    private static XhtmlHelper helper = new XhtmlHelper();


    // -------------------------------------------------------- Renderer Methods


    /**
     * <p>Decode the submitted value for this component.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param component <code>UIComponent</code> being decoded
     */
    public void decode(FacesContext context, UIComponent component) {

        if ((context == null) || (component == null)) {
            throw new NullPointerException();
        }

        AutoCompleteComponent auto = (AutoCompleteComponent) component;
        if (auto.isDisabled() || auto.isReadonly()) {
            return;
        }
        String clientId = auto.getClientId(context);
        String submittedValue = (String)
                context.getExternalContext().getRequestParameterMap().get(clientId);
//        System.out.println("decode(" + clientId + "," + submittedValue + ")");
        auto.setSubmittedValue(submittedValue);

    }



    /**
     * <p>Render the HTML for this component.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param component <code>UIComponent</code> being rendered
     *
     * @exception IOException if an input/output error occurs
     */
    public void encodeEnd(FacesContext context, UIComponent component)
        throws IOException {

        // Prepare the local variables we will need
        AutoCompleteComponent auto = (AutoCompleteComponent) component;
        String clientId = auto.getClientId(context);
        String name = name(context, auto);
        ResponseWriter writer = context.getResponseWriter();

        // Render the required resource links once per page (runtime only)
        if (!Beans.isDesignTime()) {
            helper.linkStylesheet(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  CSS_RESOURCE);
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  Util.UI_COMMON_SCRIPT_RESOURCE);
            Util.renderDojoLoading(context, component, writer);
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  SCRIPT_RESOURCE);
            // Localize the error messages for this component
            writer.startElement("script", auto);
            writer.writeAttribute("type", "text/javascript", null);
            writer.write("\n");
            writer.write("bpui.autocomplete.messages[\"bindError\"] = \"" +
                         Util.getMessage("autocomplete.bindError") + "\";");
            writer.write("\n");
            writer.endElement("script");
            writer.write("\n");
        }

        // Render a <div> for the pop-up element that will be dynamically created
        // (but only at runtime)
        if (!Beans.isDesignTime()) {
            writer.startElement("div", auto);
            writer.writeAttribute("id", clientId + "_menu", null);
            writer.writeAttribute("style",
                "position: absolute; top:170px;left:140px;visibility:hidden", null);
            writer.writeAttribute("class", "popupFrame", null);
            writer.endElement("div");
            writer.write("\n");
        }

        // Render the beginning of the <input> element for this text field
        writer.startElement("input", auto);
        writer.writeAttribute("id", clientId, "id");
        writer.writeAttribute("type", "text", null);
        writer.writeAttribute("name", clientId, "clientId");

        // Render the current value (if any)
        Object value = auto.getValue();
        if (value != null) {
            writer.writeAttribute("value", value, "value");
        }

        // Render the CSS style class(es) (if any)
        String styleClass = auto.getStyleClass();
        if (styleClass != null) {
            writer.writeAttribute("class", styleClass, "styleClass");
        }

        // Set the autocomplete attribute to "off" to disable browser 
        // textfield completion with previous entries
        writer.writeAttribute("autocomplete", "off", null);

        // Render HTML attributes that are simply passed through
        renderBooleanPassThruAttributes(writer, component);
        renderIntegerPassThruAttributes(writer, component);
        renderStringPassThruAttributes(writer, component);

        // Emit the javascript for auto completion (if a completion method
        // has been declared)
        if (auto.getCompletionMethod() != null) {
            String continueScript = name + ".again()";
            String startScript = name + ".start()";
            String stopScript = name + ".stop()";
            writer.writeAttribute("onfocus", startScript, null);
            writer.writeAttribute("onkeyup", continueScript, null);
            writer.writeAttribute("onblur", stopScript, null);
        }

        // Finish up the rendering of this component
        writer.endElement("input");
        writer.write("\n");

        // Create a JavaScript object representing this component
        if (!Beans.isDesignTime()) {
            String methodName = null;
            if (auto.getCompletionMethod() != null) {
                methodName = auto.getCompletionMethod().getExpressionString();
            }
            writer.startElement("script", auto);
            writer.writeAttribute("type", "text/javascript", null);
            writer.write("\n");
            writer.write("var " + name + " = new bpui.autocomplete.Component(");
            writer.write("dojo.byId('" + clientId + "'), ");
            writer.write("dojo.byId('" + clientId + "_menu'), ");
            if (methodName != null) {
                writer.write("'" + methodName + "', ");
            } else {
                writer.write("null, ");
            }
            writer.write("'" + helper.mapResourceId(context,
                                                    Mechanism.DYNAMIC_RESOURCE,
                                                    COMPLETION_HANDLER) + "', ");
            if (auto.getOnchoose() != null) {
                writer.write(auto.getOnchoose() + ", ");
            } else {
                writer.write("null, ");
            }
            if (auto.getOndisplay() != null) {
                writer.write(auto.getOndisplay());
            } else {
                writer.write("null");
            }
            writer.write(");");
            writer.write("\n");
            writer.endElement("script");
            writer.write("\n");
        }

    }


    // ------------------------------------------------------==- Private Methods


    /**
     * <p>Return the name of a JavaScript variable we can use to create and
     * manipulate the client side object for the specified component.
     * This name must be unique across all components instantiated on
     * the same page.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param auto {@link AutoCompleteComponent} being rendered
     */
    private String name(FacesContext context, AutoCompleteComponent auto) {

        return auto.getClientId(context).replace(':', '_');

    }


    /**
     * <p>Render pass-through HTML attributes with boolean values.</p>
     *
     * @param writer <code>ResponseWriter</code> to render with
     * @param component <code>UIComponent</code> being rendered
     */
    private void renderBooleanPassThruAttributes(ResponseWriter writer,
            UIComponent component) throws IOException {

        boolean result = false;
        Object value;
        for (int i = 0; i < booleanPassthruAttributes.length; i++) {
            value = component.getAttributes().get(booleanPassthruAttributes[i]);
            if (value != null) {
                if (value instanceof Boolean) {
                    result = ((Boolean) value).booleanValue();
                } else if (value instanceof String) {
                    result = (new Boolean((String) value)).booleanValue();
                }
                if (result) {
                    writer.writeAttribute(booleanPassthruAttributes[i],
                                          booleanPassthruAttributes[i],
                                          booleanPassthruAttributes[i]);
                }
            }
        }

    }



    /**
     * <p>Render pass-through HTML attributes with integer values.</p>
     *
     * @param writer <code>ResponseWriter</code> to render with
     * @param component <code>UIComponent</code> being rendered
     */
    private void renderIntegerPassThruAttributes(ResponseWriter writer,
            UIComponent component) throws IOException {

        Object value;
        for (int i = 0; i < integerPassthruAttributes.length; i++) {
            value = component.getAttributes().get(integerPassthruAttributes[i]);
            if ((value != null) && (value instanceof Integer)) {
                int intValue = ((Integer) value).intValue();
                if (intValue != Integer.MIN_VALUE) {
                    writer.writeAttribute(integerPassthruAttributes[i],
                                          value.toString(),
                                          integerPassthruAttributes[i]);
                }
            }
        }

    }


    /**
     * <p>Render pass-through HTML attributes with String values.</p>
     *
     * @param writer <code>ResponseWriter</code> to render with
     * @param component <code>UIComponent</code> being rendered
     */
    private void renderStringPassThruAttributes(ResponseWriter writer,
            UIComponent component) throws IOException {

        Object value;
        for (int i = 0; i < passthruAttributes.length; i++) {
            value = component.getAttributes().get(passthruAttributes[i]);
            if (value != null) {
                if (!(value instanceof String)) {
                    value = value.toString();
                }
                writer.writeAttribute(passthruAttributes[i],
                                      (String) value,
                                      passthruAttributes[i]);
            }
        }

    }


}
