/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: RichTextareaEditorRenderer.java,v 1.2 2006/04/26 04:03:01 craig_mcc Exp $
 */

package com.sun.j2ee.blueprints.ui.editor;

import com.sun.j2ee.blueprints.ui.util.Util;
import java.beans.Beans;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.faces.application.ViewHandler;
import javax.faces.component.UIComponent;
import javax.faces.component.UIViewRoot;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.Renderer;
import org.apache.shale.remoting.Mechanism;
import org.apache.shale.remoting.XhtmlHelper;

/**
 * <p>This renderer emits an HTML <code>textarea</code> element, but instruments
 * it with rich text editing facilities using the "Editor" widget in the
 * DOJO toolkit.</p>
 */
public class RichTextareaEditorRenderer extends Renderer {


    // ------------------------------------------------------ Manifest Constants


    /**
     * <p>The static script resource for this component.</p>
     */
    private static final String SCRIPT_RESOURCE =
            "/META-INF/editor/script.js";


    // -------------------------------------------------------- Static Variables


    /**
     * <p>HTML attributes with boolean values that are passed through
     * to the rendered output.</p>
     */
    private static String booleanPassthruAttributes[] =
    { "disabled", "readonly", "ismap" };


    /**
     * <p>HTML attributes with integer values that are passed through
     * to the rendered output.</p>
     */
    private static String integerPassthruAttributes[] =
    { "cols", "maxlength", "rows", "size" };


    /**
     * <p>HTML attributes with String values that are passed through
     * to the rendered output.</p>
     */
    private static String passthruAttributes[] =
    { "accept", "accesskey", "alt", "bgcolor", "border",
      "cellpadding", "cellspacing", "charset",
      "coords", "dir", "enctype", "frame", "height",
      "hreflang", "lang", "longdesc", "onblur",
      "onchange", "onclick", "ondblclick", "onfocus",
      "onkeydown", "onkeypress", "onkeyup", "onload",
      "onmousedown", "onmousemove", "onmouseout",
      "onmouseover", "onmouseup", "onreset", "onselect",
      "onsubmit", "onunload", "rel", "rev",
      "rules", "shape", "summary",
      "tabindex", "target", "title", "usemap", "width" };


    /**
     * <p>Stateless helper bean to manufacture resource linkages.</p>
     */
    private static XhtmlHelper helper = new XhtmlHelper();


    // -------------------------------------------------------- Renderer Methods


    /**
     * <p>Decode the submitted value for this component.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param component <code>UIComponent</code> being decoded
     */
    public void decode(FacesContext context, UIComponent component) {

        if ((context == null) || (component == null)) {
            throw new NullPointerException();
        }

        RichTextareaEditorComponent editor = (RichTextareaEditorComponent) component;
        if (editor.isDisabled() || editor.isReadonly()) {
            return;
        }
        String clientId = editor.getClientId(context);
        String submittedValue = (String)
                context.getExternalContext().getRequestParameterMap().get(clientId);
//        System.out.println("decode(" + clientId + "," + submittedValue + ")");
        editor.setSubmittedValue(submittedValue);

    }



    /**
     * <p>Render the HTML for this component.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param component <code>UIComponent</code> being rendered
     *
     * @exception IOException if an input/output error occurs
     */
    public void encodeEnd(FacesContext context, UIComponent component)
        throws IOException {

        // Prepare the local variables we will need
        RichTextareaEditorComponent editor = (RichTextareaEditorComponent) component;
        String clientId = editor.getClientId(context);
        ResponseWriter writer = context.getResponseWriter();

        // Render the required resource links once per page (runtime only)
        if (!Beans.isDesignTime()) {
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  Util.UI_COMMON_SCRIPT_RESOURCE);
            Util.renderDojoLoading(context, component, writer);
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  SCRIPT_RESOURCE);
            // Localize the error messages for this component
            /*
            writer.startElement("script", editor);
            writer.writeAttribute("type", "text/javascript", null);
            writer.write("\n");
            writer.write("bpui.editor.messages[\"foo\"] = \"" +
                         Util.getMessage("editor.foo") + "\";");
            writer.write("\n");
            writer.endElement("script");
            writer.write("\n");
            */
        }

        // Render an outer DIV element to contain this widget
        writer.startElement("div", editor);
        String style = editor.getStyle();
        if (style != null) {
            // FIXME - first kludge for solid border
            if (style.indexOf("border") < 1) {
                style += "; border: 1px solid black";
            }
            writer.writeAttribute("style", style, "style");
        } else {
            // FIXME - second kludge for solid border
            writer.writeAttribute("style", "border: 1px solid black", null);
        }
        String styleClass = editor.getStyleClass();
        if (styleClass != null) {
            writer.writeAttribute("class", styleClass, "styleClass");
        }
        writer.write("\n");

        // Render a <textarea> that will contain the editable text, and
        // instrument it with the DOJO Editor widget
        writer.startElement("textarea", editor);
        writer.writeAttribute("id", clientId, "id");
        writer.writeAttribute("name", clientId, "clientId");
        writer.writeAttribute("dojoType", "Editor", null);
        if (Beans.isDesignTime()) {
            writer.writeAttribute("style", "height: 100%; width: 100%", null);
        }
        String items = editor.getItems();
        if (items != null) {
            writer.writeAttribute("items", items, "items");
        }
        renderBooleanPassThruAttributes(writer, component);
        renderIntegerPassThruAttributes(writer, component);
        renderStringPassThruAttributes(writer, component);
        writer.write("\n");
        Object value = editor.getValue();
        if (value != null) {
            writer.write(value.toString());
            writer.write("\n");
        }
        writer.endElement("textarea");
        writer.write("\n");
        writer.endElement("div");
        writer.write("\n");

    }


    // ------------------------------------------------------==- Private Methods


    /**
     * <p>Render pass-through HTML attributes with boolean values.</p>
     *
     * @param writer <code>ResponseWriter</code> to render with
     * @param component <code>UIComponent</code> being rendered
     */
    private void renderBooleanPassThruAttributes(ResponseWriter writer,
            UIComponent component) throws IOException {

        boolean result = false;
        Object value;
        for (int i = 0; i < booleanPassthruAttributes.length; i++) {
            value = component.getAttributes().get(booleanPassthruAttributes[i]);
            if (value != null) {
                if (value instanceof Boolean) {
                    result = ((Boolean) value).booleanValue();
                } else if (value instanceof String) {
                    result = (new Boolean((String) value)).booleanValue();
                }
                if (result) {
                    writer.writeAttribute(booleanPassthruAttributes[i],
                                          booleanPassthruAttributes[i],
                                          booleanPassthruAttributes[i]);
                }
            }
        }

    }



    /**
     * <p>Render pass-through HTML attributes with integer values.</p>
     *
     * @param writer <code>ResponseWriter</code> to render with
     * @param component <code>UIComponent</code> being rendered
     */
    private void renderIntegerPassThruAttributes(ResponseWriter writer,
            UIComponent component) throws IOException {

        Object value;
        for (int i = 0; i < integerPassthruAttributes.length; i++) {
            value = component.getAttributes().get(integerPassthruAttributes[i]);
            if ((value != null) && (value instanceof Integer)) {
                int intValue = ((Integer) value).intValue();
                if (intValue != Integer.MIN_VALUE) {
                    writer.writeAttribute(integerPassthruAttributes[i],
                                          value.toString(),
                                          integerPassthruAttributes[i]);
                }
            }
        }

    }


    /**
     * <p>Render pass-through HTML attributes with String values.</p>
     *
     * @param writer <code>ResponseWriter</code> to render with
     * @param component <code>UIComponent</code> being rendered
     */
    private void renderStringPassThruAttributes(ResponseWriter writer,
            UIComponent component) throws IOException {

        Object value;
        for (int i = 0; i < passthruAttributes.length; i++) {
            value = component.getAttributes().get(passthruAttributes[i]);
            if (value != null) {
                if (!(value instanceof String)) {
                    value = value.toString();
                }
                writer.writeAttribute(passthruAttributes[i],
                                      (String) value,
                                      passthruAttributes[i]);
            }
        }

    }


}
