/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: MapRenderer.java,v 1.18 2006/03/06 18:51:40 basler Exp $
 */

package com.sun.j2ee.blueprints.ui.mapviewer;

import com.sun.j2ee.blueprints.ui.util.Util;
import java.beans.Beans;
import java.io.IOException;
import javax.faces.application.ViewHandler;
import javax.faces.component.UIComponent;
import javax.faces.component.UIViewRoot;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.el.ValueBinding;
import javax.faces.render.Renderer;
import org.apache.shale.remoting.Mechanism;
import org.apache.shale.remoting.XhtmlHelper;

/**
 * <p>Renderer for the {@link MapComponent} component.  For documentation on
 * Google Maps APIs, see <a href="http://www.google.com/apis/maps/">
 * http://www.google.com/apis/maps/</a>.</p>
 */
public class MapRenderer extends Renderer {
    

    // ------------------------------------------------------ Manifest Constants


    /**
     * <p>Resource name of the background image that will be rendered
     * at design time only.</p>
     */
    private static final String BACKGROUND_IMAGE =
            "/com/sun/j2ee/blueprints/ui/mapviewer/Background.jpg";


    /**
     * <p>Request scope attribute we use to denote the fact that the
     * Google Maps APIs javascript has already been loaded for this page.
     */
    private static final String GOOGLE_LOADED =
            "com.sun.j2ee.mapviewer.GOOGLE_LOADED";


    /**
     * <p>The prefix of the static script resource for the Google Maps APIs.
     * Append the authorization key to this before submitting.</p>
     */
    private static final String GOOGLE_RESOURCE =
            "http://maps.google.com/maps?file=api&v=1&key=";


    /**
     * <p>The static script resource for this component.</p>
     */
    private static final String SCRIPT_RESOURCE =
            "/META-INF/mapviewer/script.js";


    /**
     * <p>The context initialization parameter consulted to discover the
     * Google Maps authentication key, if not present on this component.</p>
     */
    private static final String AUTHORIZATION_KEY =
            "com.sun.j2ee.blueprints.ui.mapviewer.KEY";


    /**
     * <p>Stateless helper bean to manufacture resource linkages.</p>
     */
    private static XhtmlHelper helper = new XhtmlHelper();


    // -------------------------------------------------------- Static Variables


    // -------------------------------------------------------- Renderer Methods


    public void encodeEnd(FacesContext context, UIComponent component)
        throws IOException {

        MapComponent map = (MapComponent) component;
        ResponseWriter writer = context.getResponseWriter();

        // Render an outer <div> around the entire output
        writer.startElement("div", map);
        writer.writeAttribute("id", map.getClientId(context), "id");
        String styleClass = map.getStyleClass();
        if (styleClass != null) {
            writer.writeAttribute("class", styleClass, "styleClass");
        }
        String style = map.getStyle();
        if (style != null) {
            writer.writeAttribute("style", style, "style");
        }
        writer.write("\n");

        // If not already done on this page, get the Google Maps JavaScript,
        // as well as the JavaScript specific to this component
        // (runtime only)
        if (!Beans.isDesignTime()) {
            if (!context.getExternalContext().getRequestMap().containsKey(GOOGLE_LOADED)) {
                writer.startElement("script", map);
                writer.writeAttribute("type", "text/javascript", null);
                writer.writeURIAttribute("src", url(context, map), null);
                writer.endElement("script");
                writer.write("\n");
                context.getExternalContext().getRequestMap().put(GOOGLE_LOADED, Boolean.TRUE);
            }
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  Util.UI_COMMON_SCRIPT_RESOURCE);
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  SCRIPT_RESOURCE);
        }

        // Render a <div> element to receive the map
        // (at design time, show interesting information about the map here)
        writer.startElement("div", map);
        writer.writeAttribute("id", map.getClientId(context) + "_map", "id");
        style = "height: 100%; width: 100%;";
        if (Beans.isDesignTime()) {
            style += " border: 1px dashed gray;" +
                     " color: black;" +
                     " text-align: center;" +
                     " vertical-align: middle;";
            String url =
                MapRenderer.class.getResource(BACKGROUND_IMAGE).toString();
            style += " background-image: url('" + url + "');" +
                     " background-repeat: repeat;";
        }
        writer.writeAttribute("style", style, null);
        if (Beans.isDesignTime()) {
            writer.startElement("p", map);
            writer.writeText(Util.getMessage("mapviewer.design.title"), null);
            writer.endElement("p");
            writer.write("\n");
            ValueBinding vb = null;
            vb = map.getValueBinding("center");
            if (vb != null) {
                writer.startElement("p", map);
                writer.writeText(Util.getMessage("mapviewer.design.center"), null);
                writer.writeText(vb.getExpressionString(), null);
                writer.endElement("p");
                writer.write("\n");
            }
            vb = map.getValueBinding("info");
            if (vb != null) {
                writer.startElement("p", map);
                writer.writeText(Util.getMessage("mapviewer.design.info"), null);
                writer.writeText(vb.getExpressionString(), null);
                writer.endElement("p");
                writer.write("\n");
            }
            vb = map.getValueBinding("markers");
            if (vb != null) {
                writer.startElement("p", map);
                writer.writeText(Util.getMessage("mapviewer.design.markers"), null);
                writer.writeText(vb.getExpressionString(), null);
                writer.endElement("p");
                writer.write("\n");
            }
        }
        writer.endElement("div");
        writer.write("\n");

        // Render the JavaScript code to retrieve and display the map
        // (runtime only)
        if (!Beans.isDesignTime()) {

            String name = name(context, map);
            String clientId = map.getClientId(context);
            double latitude = 0.0;
            double longitude = 0.0;
            MapPoint point = map.getCenter();
            if (point != null) {
                latitude = point.getLatitude();
                longitude = point.getLongitude();
            }
            if ((latitude == 0.0) && (longitude == 0.0)) {
                latitude = 37.4419;
                longitude = -122.1419;
            }
            int zoomLevel = map.getZoomLevel();
            if (zoomLevel < 1) {
                zoomLevel = 4;
            }

            writer.startElement("script", map);
            writer.writeAttribute("type", "text/javascript", null);
            writer.write("\n");

            // Create a global variable pointing to the map
            writer.writeText("var " + name + " = null;\n", null);

            // Begin generating a setup function for this map
            writer.writeText("bpui.mapviewer.setup_" + name + " = function() {\n", null);

            // Create and configure the map and its controls
            writer.writeText("  " + name + " = bpui.mapviewer.createMap(" +
                             "\"" + clientId + "_map\", " +
                             latitude + ", " + longitude + ", " + zoomLevel +
                             ");\n", null);
            if (!map.isDraggingEnabled()) {
                writer.writeText("  bpui.mapviewer.setDragging(" + name + ", false);\n", null);
            }
            if (map.isMapControls()) {
                writer.writeText("  bpui.mapviewer.addControl(" + name + ", " +
                                 "bpui.mapviewer.createMapControl());\n", null);
            }
            if (map.isTypeControls()) {
                writer.writeText("  bpui.mapviewer.addControl(" + name + ", " +
                                 "bpui.mapviewer.createTypeControl());\n", null);
            }

            // Set up the info window, if specified
            MapMarker info = map.getInfo();
            if ((info != null) && (info.getMarkup() != null)) {
                if ((info.getLatitude() != 0.0) && (info.getLongitude() != 0.0)) {
                    writer.writeText("  bpui.mapviewer.openInfoWindowHtml(" + name + ", " +
                            info.getLatitude() + ", " + info.getLongitude() + ", " +
                            "\"" + markup(info.getMarkup()) + "\"" + ");\n", null);
                } else {
                    writer.writeText("  bpui.mapviewer.openInfoWindowHtml(" + name + ", " +
                            latitude + ", " + longitude + ", " +
                            "\"" + markup(info.getMarkup()) + "\"" + ");\n", null);
                }
            }

            // Set up the markers, if specified
            MapMarker markers[] = map.getMarkers();
            if (markers != null) {
                for (int i = 0; i < markers.length; i++) {
                    MapMarker marker = markers[i];
                    if (marker.getMarkup() != null) {
                        writer.writeText("  bpui.mapviewer.addMarker(" + name + ", " +
                                         marker.getLatitude() + ", " +
                                         marker.getLongitude() + ", " +
                                         "\"" + markup(marker.getMarkup()) + "\"" +
                                         ");\n", null);
                    } else {
                        writer.writeText("  bpcui_mapviewer_addMarker(" + name + ", " +
                                         marker.getLatitude() + ", " +
                                         marker.getLongitude() + ", " +
                                         "null" +
                                         ");\n", null);
                    }
                }
            }

            // Finish the setup function
            writer.writeText("}\n", null);

            // Call the setup function directly
            writer.writeText("setTimeout('bpui.mapviewer.setup_" + name +
                             "()', 0);\n", null);

            writer.endElement("script");
            writer.write("\n");

        }

        // Close the outer <div> element
        writer.endElement("div");
        writer.write("\n");

    }


    // --------------------------------------------------------- Private Methods


    /**
     * <p>Return the specified HTML markup string, with any embedded double
     * quote characters escaped.</p>
     *
     * @param markup Markup string to be encoded
     */
    private String markup(String markup) {
        if (markup.indexOf('"') < 0) {
            return markup;
        }
        StringBuffer sb = new StringBuffer(markup.length() + 4);
        for (int i = 0; i < markup.length(); i++) {
            char ch = markup.charAt(i);
            if (ch == '"') {
                sb.append('\\');
            }
            sb.append(ch);
        }
        return sb.toString();
    }


    /**
     * <p>Return the name of a JavaScript variable we can use to create and
     * manipulate the map for the specified component.  This name must be
     * unique across all map components instantiated on the same page.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param map {@link MapComponent} being rendered
     */
    private String name(FacesContext context, MapComponent map) {

        return map.getClientId(context).replace(':', '_');

    }


    /**
     * <p>Return a "new GPoint()" expression encapsulating the latitude and
     * longitude from the specified {@link MapPoint}.</p>
     *
     * @param point {@link MapPoint} to be encapsulated
     */
    private String point(MapPoint point) {
        return "new GPoint(" + point.getLongitude() + ", " + point.getLatitude() + ")";
    }


    /**
     * <p>Compose and return the URL to retrieve the map for this component
     *
     * @param context <code>FacesContext</code> for this request
     * @param map {@link MapComponent} being rendered
     */
    private String url(FacesContext context, MapComponent map) {

        StringBuffer sb = new StringBuffer(GOOGLE_RESOURCE);
        String key = map.getKey();
        if (key == null) {
            key = context.getExternalContext().getInitParameter(AUTHORIZATION_KEY);
        }
        sb.append(key);
        return sb.toString();

    }


}
