/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: PopupCalendarComponent.java,v 1.6 2006/06/05 22:14:41 edwingo Exp $
 */

package com.sun.j2ee.blueprints.ui.popupcalendar;

import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.DecimalFormat;
import java.text.MessageFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.faces.application.FacesMessage;
import javax.faces.component.NamingContainer;
import javax.faces.component.html.HtmlInputText;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.ConverterException;
import javax.faces.el.ValueBinding;

import com.sun.j2ee.blueprints.ui.util.Util;

/**
 * Date picker with a JavaScript popup calendar
 * 
 * @author Edwin Goei
 */
public class PopupCalendarComponent extends HtmlInputText {
    private static final String LOCALE_STRING = "localeString";

    private static final String DATE_FORMAT_PATTERN = "dateFormatPattern";

    private static final String FIELD_SUFFIX = "_field";

    /** ISO 8601 date format pattern */
    private static final String ISO_DATE_PATTERN = "yyyy-MM-dd";

    private static final String ISO_DATE_SEPARATOR = "-";

    private static final DecimalFormat DECIMAL_FORMAT4 = new DecimalFormat(
            "0000");

    private static final DecimalFormat DECIMAL_FORMAT2 = new DecimalFormat("00");

    /** DateFormat corresponding to ISO pattern */
    private SimpleDateFormat isoDateFormat;

    private SimpleDateFormat localizedDateFormat;

    private String localizedPrompt;

    /** SimpleDateFormat says [a-zA-Z] are reserved. */
    private static final Pattern separatorPattern = Pattern
            .compile("[^a-zA-Z]");

    /** Javascript first day of week, Sunday = 0, Monday = 1, ... */
    private int jsFirstDayOfWeek;

    private String localeString = null;

    private boolean fieldsInitialized;

    /** Information on date format pattern */
    private PatternInfo patternInfo;

    /**
     * Contains info on a date format pattern
     */
    private static class PatternInfo {
        private int yearIndex;

        private int monthIndex;

        private int dayIndex;

        private String separator;

        public PatternInfo(int yearIndex, int monthIndex, int dayIndex,
                String separator) {
            this.yearIndex = yearIndex;
            this.monthIndex = monthIndex;
            this.dayIndex = dayIndex;
            this.separator = separator;
        }

        public int getDayIndex() {
            return dayIndex;
        }

        public int getMonthIndex() {
            return monthIndex;
        }

        public String getSeparator() {
            return separator;
        }

        public int getYearIndex() {
            return yearIndex;
        }
    }

    public PopupCalendarComponent() {
        super();
        setRendererType("PopupCalendar");
    }

    public String getFamily() {
        return "PopupCalendar";
    }

    /**
     * Call this method to refresh fields that depend on properties of this
     * component. See fieldsInitialized flag.
     */
    private void ensureDerivedFields() {
        if (fieldsInitialized) {
            return;
        }

        Locale locale;
        String localeString = getLocaleString();
        if (localeString != null) {
            String[] parts = localeString.split("[-_]", 3);
            switch (parts.length) {
            case 3:
                locale = new Locale(parts[0], parts[1], parts[2]);
                break;
            case 2:
                locale = new Locale(parts[0], parts[1]);
                break;
            case 1:
            default:
                locale = new Locale(parts[0]);
                break;
            }
        } else {
            locale = getFacesContext().getViewRoot().getLocale();
        }

        initLocaleDependentFields(locale);
        fieldsInitialized = true;
    }

    private void initLocaleDependentFields(Locale locale) {
        // Init a localized ISO date format (just to be safe)
        isoDateFormat = new SimpleDateFormat(ISO_DATE_PATTERN, locale);

        SimpleDateFormat simpleDateFormat;
        String pattern = getDateFormatPattern();
        if (pattern != null) {
            simpleDateFormat = new SimpleDateFormat(pattern, locale);
        } else {
            // Short date format should be all numeric
            DateFormat dateFormat = DateFormat.getDateInstance(
                    DateFormat.SHORT, locale);
            if (dateFormat instanceof SimpleDateFormat) {
                simpleDateFormat = (SimpleDateFormat) dateFormat;
            } else {
                // Javadoc claims this code path is possible (but unlikely) so
                // we initialize a localized ISO date format above to be safe

                // Default to ISO format
                simpleDateFormat = isoDateFormat;
            }
            pattern = simpleDateFormat.toPattern();
        }

        // Validate the pattern
        int yearIndex = pattern.indexOf('y');
        int monthIndex = pattern.indexOf('M');
        int dayIndex = pattern.indexOf('d');
        if (yearIndex == -1 || monthIndex == -1 || dayIndex == -1) {
            // Pattern does not contain all date parts so default to ISO format
            pattern = ISO_DATE_PATTERN;
            yearIndex = pattern.indexOf('y');
            monthIndex = pattern.indexOf('M');
            dayIndex = pattern.indexOf('d');
        }

        // Find the separator or use a default
        Matcher mat = separatorPattern.matcher(pattern);
        String separator = mat.find() ? mat.group() : ISO_DATE_SEPARATOR;

        // Compute the localized pattern strings
        String localPatternChars = simpleDateFormat.getDateFormatSymbols()
                .getLocalPatternChars();
        char yearChar = localPatternChars.charAt(DateFormat.YEAR_FIELD);
        String year = String.valueOf(new char[] { yearChar, yearChar, yearChar,
                yearChar });
        char monthChar = localPatternChars.charAt(DateFormat.MONTH_FIELD);
        String month = String.valueOf(new char[] { monthChar, monthChar });
        char dayChar = localPatternChars.charAt(DateFormat.DATE_FIELD);
        String day = String.valueOf(new char[] { dayChar, dayChar });

        // Handles most common ordering of fields: year is first or last
        StringBuffer prompt = new StringBuffer();
        StringBuffer newPattern = new StringBuffer();
        if (yearIndex < dayIndex && yearIndex < monthIndex) {
            // Year is first
            prompt.append(year);
            prompt.append(separator);

            newPattern.append("yyyy");
            newPattern.append(separator);
        }
        if (dayIndex < monthIndex) {
            prompt.append(day);
            prompt.append(separator);
            prompt.append(month);

            newPattern.append("dd");
            newPattern.append(separator);
            newPattern.append("MM");
        } else {
            prompt.append(month);
            prompt.append(separator);
            prompt.append(day);

            newPattern.append("MM");
            newPattern.append(separator);
            newPattern.append("dd");
        }
        if (yearIndex > dayIndex && yearIndex > monthIndex) {
            // Year is last
            prompt.append(separator);
            prompt.append(year);

            newPattern.append(separator);
            newPattern.append("yyyy");
        }

        // Recompute pattern info based on new pattern
        String newPatternStr = newPattern.toString();
        yearIndex = newPatternStr.indexOf('y');
        monthIndex = newPatternStr.indexOf('M');
        dayIndex = newPatternStr.indexOf('d');
        this.patternInfo = new PatternInfo(yearIndex, monthIndex, dayIndex,
                separator);

        this.localizedDateFormat = new SimpleDateFormat(newPatternStr, locale);
        this.localizedPrompt = prompt.toString().toLowerCase(locale);

        Calendar cal = Calendar.getInstance(locale);
        // Javascript uses 0 to mean Sunday
        this.jsFirstDayOfWeek = cal.getFirstDayOfWeek() - Calendar.SUNDAY;
    }

    /**
     * Method used by the renderer
     * 
     * @return text to be displayed in the input field
     */
    String getInputTextValue() {
        // If no selected date, then use the localized prompt pattern
        String selectedDate = getFormattedModelValue();
        if (selectedDate == null) {
            return getLocalizedPrompt();
        } else {
            return selectedDate;
        }
    }

    private String getFormattedModelValue() {
        FacesContext context = FacesContext.getCurrentInstance();

        Object value = getValue();
        if (value == null) {
            return null;
        }

        String isoDate = null;

        // Try to convert the model value to an ISO date value
        Converter converter = getConverter();
        if (converter != null) {
            // There is a converter so convert model value to ISO date using
            // converter
            try {
                isoDate = converter.getAsString(context, this, value);
            } catch (ConverterException e) {
                return null;
            }
        } else {
            // No converter
            if (value instanceof String) {
                // Assume that value is already in ISO date format
                isoDate = (String) value;
            } else {
                // Try to automatically convert the model value to ISO date
                converter = new SqlUtilDateConverter();
                try {
                    isoDate = converter.getAsString(context, this, value);
                } catch (ConverterException e) {
                    return null;
                }
                setConverter(converter);
            }
        }

        ensureDerivedFields();
        // Convert the ISO date into a localized format
        Date date;
        try {
            date = isoDateFormat.parse(isoDate);
        } catch (ParseException e) {
            // Unable to parse value so act like no date has been selected
            return null;
        }

        // Convert to a localized pattern. This is the normal return path.
        return localizedDateFormat.format(date);
    }

    /**
     * For use by renderer
     * 
     * @return tooltip to be rendered
     */
    String getInputTextTooltip() {
        return getLocalizedPrompt();
    }

    /**
     * For use by renderer
     * 
     * @return unique client id for the input text sub component
     */
    String getInputTextClientId(FacesContext context) {
        return getJavaScriptObjectName(context) + FIELD_SUFFIX;
    }

    /**
     * @return unique name used as a javascript object identifier
     */
    String getJavaScriptObjectName(FacesContext context) {
        return getClientId(context)
                .replace(NamingContainer.SEPARATOR_CHAR, '_');
    }

    /**
     * For use by renderer
     * 
     * @param context
     * @return javascript code for a new instance of this calendar
     */
    String getJavaScriptForNewInstance(FacesContext context) {
        ensureDerivedFields();
        StringBuffer js = new StringBuffer("var "
                + getJavaScriptObjectName(context));
        js.append(" = dojo.widget.createWidget('PopupCalendar', { baseId: '");
        js.append(getJavaScriptObjectName(context) + "',\n");

        js.append("calendarParams: { yearIndex: " + patternInfo.getYearIndex());
        js.append(", monthIndex: " + patternInfo.getMonthIndex());
        js.append(", dayIndex: " + patternInfo.getDayIndex());
        js.append(", separator: '" + patternInfo.getSeparator() + "',\n");

        js.append("months: ['");
        DateFormatSymbols dfs = localizedDateFormat.getDateFormatSymbols();
        String[] months = dfs.getMonths();
        for (int i = 0; i < 11; i++) {
            js.append(months[i]);
            js.append("', '");
        }
        js.append(months[11]);
        js.append("'],\n");

        // Use Javascript conventions of Sunday = 0
        js.append("weekdays: ['");
        String[] weekdays = dfs.getShortWeekdays();
        for (int i = Calendar.SUNDAY; i < Calendar.SATURDAY; i++) {
            js.append(weekdays[i]);
            js.append("', '");
        }
        js.append(weekdays[Calendar.SATURDAY]);
        js.append("'],\n");

        js.append("jsFirstWeekday: " + jsFirstDayOfWeek + " }});\n");

        return js.toString();
    }

    // @Override
    public void validate(FacesContext context) {
        // This method transforms the "submitted value" which is the request
        // parameter value into a local value. This occurs in two steps: convert
        // submitted value to an ISO date format String and then convert that to
        // the local value type which is the same as the model value type.
        // After this method executes, one of the following will be true:
        // 1) error condition with a queued error message and valid=false
        // 2) submitted value = "" to mean no date selected
        // 3) submitted value in ISO date format, meaning date was selected
        // In case 2 and 3, super.validate() will convert to an appropriate
        // local value via getConvertedValue().

        String submittedValue = (String) getSubmittedValue();
        if (submittedValue == null) {
            // I think this means that client did not submit a value
            return;
        }

        // If text is initial prompt then this is not an error
        if (Pattern.matches("^" + getLocalizedPrompt() + "$", submittedValue)) {
            setSubmittedValue("");
            super.validate(context);
            return;
        }

        // Check for only whitespace chars which is not an error
        Pattern pat = Pattern.compile("\\S");
        Matcher matcher = pat.matcher(submittedValue);
        if (!matcher.find()) {
            setSubmittedValue("");
            super.validate(context);
            return;
        }

        Pattern pat2 = Pattern.compile("\\d+");
        Matcher matcher2 = pat2.matcher(submittedValue);
        ArrayList numStrings = new ArrayList(3);
        int i = 0;
        while (matcher2.find() && i < 3) {
            numStrings.add(matcher2.group());
            i++;
        }
        // Check that we have 3 separated numbers
        if (i != 3) {
            errorBadFormat(context);
            return;
        }

        String yearString;
        ensureDerivedFields();
        int dayIndex = patternInfo.getDayIndex();
        int monthIndex = patternInfo.getMonthIndex();
        int yearIndex = patternInfo.getYearIndex();
        if (yearIndex < dayIndex && yearIndex < monthIndex) {
            // Year is first
            yearString = (String) numStrings.remove(0);
        } else {
            // Assume year is last
            yearString = (String) numStrings.remove(2);
        }

        StringBuffer isoString = new StringBuffer();
        int year = Integer.parseInt(yearString);
        isoString.append(zeroPad(year, 4));
        isoString.append(ISO_DATE_SEPARATOR);

        String monthString;
        String dayString;
        if (dayIndex < monthIndex) {
            dayString = (String) numStrings.get(0);
            monthString = (String) numStrings.get(1);
        } else {
            monthString = (String) numStrings.get(0);
            dayString = (String) numStrings.get(1);
        }
        int month = Integer.parseInt(monthString);
        int day = Integer.parseInt(dayString);
        isoString.append(zeroPad(month, 2));
        isoString.append(ISO_DATE_SEPARATOR);
        isoString.append(zeroPad(day, 2));
        setSubmittedValue(isoString.toString());
        super.validate(context);
    }

    // @Override
    protected Object getConvertedValue(FacesContext context, Object isoValue)
            throws ConverterException {
        if (isoValue instanceof String) {
            // If we have a converter, perform a conversion from ISO date String
            // to local value type
            Converter converter = getConverterWithType(context);
            if (converter != null) {
                return converter.getAsObject(context, this, (String) isoValue);
            } else if ("".equals(isoValue)) {
                // Convert "" to a null local/model value type meaning that
                // no date was selected
                return null;
            }
        } else {
            // TODO this statement causes javadoc errors on JDK 1.4
            // assert false;
        }
        return isoValue;
    }

    private Converter getConverterWithType(FacesContext context) {
        Converter converter = getConverter();
        if (converter != null) {
            return converter;
        }

        // Try to automatically set a converter if needed
        ValueBinding vb = getValueBinding("value");
        if (vb != null) {
            Class type = vb.getType(context);
            if (Date.class.isAssignableFrom(type)) {
                converter = new SqlUtilDateConverter();
                setConverter(converter);
                return converter;
            }
        }

        // Assume model value is String in ISO date format
        return null;
    }

    private String zeroPad(int number, int width) {
        DecimalFormat formatter;
        if (width == 2) {
            formatter = DECIMAL_FORMAT2;
        } else if (width == 4) {
            formatter = DECIMAL_FORMAT4;
        } else {
            throw new IllegalArgumentException("Width must be 2 or 4");
        }
        return formatter.format(number);
    }

    private void errorBadFormat(FacesContext context) {
        String pattern = Util.getMessage("popupcalendar.badFormat");
        String summary = MessageFormat.format(pattern,
                new Object[] { getLocalizedPrompt() });
        FacesMessage message = new FacesMessage(summary);
        message.setSeverity(FacesMessage.SEVERITY_ERROR);
        context.addMessage(getClientId(context), message);
        setValid(false);
        setSubmittedValue(null);
        return;
    }

    /**
     * <p>
     * Underscore or dash separated locale string used to determine calendar
     * format such as year, month, date ordering, month names, and week names.
     * If null, then the default locale from the view root will be used. For
     * example, "de_DE", "fr_CA", "es".
     * </p>
     */
    public String getLocaleString() {
        if (this.localeString != null) {
            return this.localeString;
        }
        ValueBinding _vb = getValueBinding(LOCALE_STRING);
        if (_vb != null) {
            return (String) _vb.getValue(getFacesContext());
        }
        return null;
    }

    /**
     * <p>
     * Underscore or dash separated locale string used to determine calendar
     * format such as year, month, date ordering, month names, and week names.
     * If null, then the default locale from the view root will be used. For
     * example, "de_DE", "fr_CA", "es".
     * </p>
     * 
     * @see #getLocaleString()
     */
    public void setLocaleString(String localeString) {
        this.localeString = localeString;
        fieldsInitialized = false;
    }

    // dateFormatPattern
    private String dateFormatPattern = null;

    /**
     * <p>
     * Pattern to use for date format. A combination of the strings "yyyy",
     * "MM", "dd", plus a separator character, with "yyyy" either first or last.
     * If null, then derive a default one from the locale. See "locale" property
     * for details. If pattern is not valid, then ISO 8601 "yyyy-MM-dd" will be
     * used. For example, "yyyy-MM-dd", "dd.MM.yyyy", "MM/dd/yyyy".
     * </p>
     */
    public String getDateFormatPattern() {
        if (this.dateFormatPattern != null) {
            return this.dateFormatPattern;
        }
        ValueBinding _vb = getValueBinding(DATE_FORMAT_PATTERN);
        if (_vb != null) {
            return (String) _vb.getValue(getFacesContext());
        }
        return null;
    }

    /**
     * <p>
     * Pattern to use for date format. A combination of the strings "yyyy",
     * "MM", "dd", plus a separator character, with "yyyy" either first or last.
     * If null, then derive a default one from the locale. See "locale" property
     * for details. If pattern is not valid, then ISO 8601 "yyyy-MM-dd" will be
     * used. For example, "yyyy-MM-dd", "dd.MM.yyyy", "MM/dd/yyyy".
     * </p>
     * 
     * @see #getDateFormatPattern()
     */
    public void setDateFormatPattern(String dateFormatPattern) {
        this.dateFormatPattern = dateFormatPattern;
        fieldsInitialized = false;
    }

    // valueAsString
    /**
     * <p>
     * This is a String-typed alias for the "value" property and is the model
     * value that will be the initially displayed date in the control and will
     * reflect changes made by the webapp user during post-back. Without any
     * converters, this property is a String in ISO 8601 YYYY-MM-DD format. A
     * value of null means no date is selected. Use the "value" property for
     * non-String types.
     * </p>
     */
    public String getValueAsString() {
        return (String) getValue();
    }

    /**
     * <p>
     * This is a String-typed alias for the "value" property and is the model
     * value that will be the initially displayed date in the control and will
     * reflect changes made by the webapp user during post-back. Without any
     * converters, this property is a String in ISO 8601 YYYY-MM-DD format. A
     * value of null means no date is selected. Use the "value" property for
     * non-String types.
     * </p>
     * 
     * @see #getValueAsString()
     */
    public void setValueAsString(String stringValue) {
        setValue(stringValue);
    }

    // style
    private String style = null;

    /**
     * <p>
     * CSS style attribute.
     * </p>
     */
    public String getStyle() {
        if (this.style != null) {
            return this.style;
        }
        ValueBinding _vb = getValueBinding("style");
        if (_vb != null) {
            return (String) _vb.getValue(getFacesContext());
        }
        return null;
    }

    /**
     * <p>
     * CSS style attribute.
     * </p>
     * 
     * @see #getStyle()
     */
    public void setStyle(String style) {
        this.style = style;
    }

    // styleClass
    private String styleClass = null;

    /**
     * <p>
     * CSS "class" attribute.
     * </p>
     */
    public String getStyleClass() {
        if (this.styleClass != null) {
            return this.styleClass;
        }
        ValueBinding _vb = getValueBinding("styleClass");
        if (_vb != null) {
            return (String) _vb.getValue(getFacesContext());
        }
        return null;
    }

    /**
     * <p>
     * CSS "class" attribute.
     * </p>
     * 
     * @see #getStyleClass()
     */
    public void setStyleClass(String styleClass) {
        this.styleClass = styleClass;
    }

    private String getLocalizedPrompt() {
        ensureDerivedFields();
        return localizedPrompt;
    }

    // @Override
    public void setValueBinding(String name, ValueBinding binding) {
        if (DATE_FORMAT_PATTERN.equals(name) || LOCALE_STRING.equals(name)) {
            fieldsInitialized = false;
        }
        super.setValueBinding(name, binding);
    }

    /**
     * <p>
     * Restore the state of this component.
     * </p>
     */
    public void restoreState(FacesContext _context, Object _state) {
        Object _values[] = (Object[]) _state;
        super.restoreState(_context, _values[0]);
        this.dateFormatPattern = (String) _values[1];
        this.localeString = (String) _values[2];
        this.style = (String) _values[3];
        this.styleClass = (String) _values[4];
    }

    /**
     * <p>
     * Save the state of this component.
     * </p>
     */
    public Object saveState(FacesContext _context) {
        Object _values[] = new Object[5];
        _values[0] = super.saveState(_context);
        _values[1] = this.dateFormatPattern;
        _values[2] = this.localeString;
        _values[3] = this.style;
        _values[4] = this.styleClass;
        return _values;
    }
}
