/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: PopupCalendarRenderer.java,v 1.4 2006/06/19 20:10:26 edwingo Exp $
 */

package com.sun.j2ee.blueprints.ui.popupcalendar;

import java.beans.Beans;
import java.io.IOException;

import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.Renderer;

import org.apache.shale.remoting.Mechanism;
import org.apache.shale.remoting.XhtmlHelper;

import com.sun.j2ee.blueprints.ui.util.Util;

/**
 * <p>
 * This renderer generates HTML (including JavaScript) for
 * PopupCalendarComponent.
 * </p>
 * 
 * @author Edwin Goei
 */
public class PopupCalendarRenderer extends Renderer {

    // ------------------------------------------------------ Manifest Constants

    /** CSS for top level text field and icon */
    private static final String POP_CAL_CSS = "/META-INF/popupcalendar/PopupCalendar.css";

    /** CSS for subcomponent calendar showing a single month */
    private static final String CALENDAR_CSS = "/META-INF/popupcalendar/Calendar.css";

    /** Dojo package file for "dojo.widget.*" */
    private static final String DOJO_WIDGET_PACKAGE = "/META-INF/dojo/src/widget/__package__.js";

    /** Dojo package file for "popupcalendar.*" */
    private static final String POP_CAL_PACKAGE = "/META-INF/popupcalendar/__package__.js";

    /** Calendar icon */
    private static final String CALENDAR_ICON = "/META-INF/popupcalendar/PopupCalendar.png";

    /** CSS class for popup button */
    private static final String POP_CAL_BUTTON_CLASS = "PopCalButton";

    /** CSS class for popup calendar input field */
    private static final String POP_CAL_INPUT_TEXT_CLASS = "PopCalInput";

    private static final String POP_CAL_MODULE_NAME = "popupcalendar";

    /** Suffix for element in markup where the calendar subcomponent will appear */
    private static final String POP_CAL_CALENDAR_SUFFIX = "_calendar";

    /** CSS class for location of subcomponent calendar showing a single month */
    private static final Object POP_CAL_CALENDAR_CLASS = "PopCalPopup";

    // -------------------------------------------------------- Static Variables

    /**
     * <p>
     * HTML attributes with String values that are passed through to the
     * rendered output.
     * </p>
     */
    private static String passthruAttributes[] = { "accept", "accesskey",
            "alt", "bgcolor", "border", "cellpadding", "cellspacing",
            "charset", "cols", "coords", "dir", "enctype", "frame", "height",
            "hreflang", "lang", "longdesc", "onblur", "onchange", "onclick",
            "ondblclick", "onfocus", "onkeydown", "onkeypress", "onkeyup",
            "onload", "onmousedown", "onmousemove", "onmouseout",
            "onmouseover", "onmouseup", "onreset", "onselect", "onsubmit",
            "onunload", "rel", "rev", "rows", "rules", "shape", "style",
            "summary", "tabindex", "target", "title", "usemap", "width" };

    /**
     * <p>
     * Stateless helper bean to manufacture resource linkages.
     * </p>
     */
    private static XhtmlHelper helper = new XhtmlHelper();

    // -------------------------------------------------------- Renderer Methods

    /**
     * <p>
     * Decode the submitted value for this component.
     * </p>
     * 
     * @param context
     *            <code>FacesContext</code> for the current request
     * @param component
     *            <code>UIComponent</code> being decoded
     */
    public void decode(FacesContext context, UIComponent component) {

        if ((context == null) || (component == null)) {
            throw new NullPointerException();
        }

        PopupCalendarComponent popCal = (PopupCalendarComponent) component;
        if (popCal.isDisabled() || popCal.isReadonly()) {
            return;
        }

        String clientId = popCal.getInputTextClientId(context);
        String submittedValue = (String) context.getExternalContext()
                .getRequestParameterMap().get(clientId);
        popCal.setSubmittedValue(submittedValue);
    }

    /**
     * <p>
     * Render the HTML for this component.
     * </p>
     * 
     * @param context
     *            <code>FacesContext</code> for the current request
     * @param component
     *            <code>UIComponent</code> being rendered
     * 
     * @exception IOException
     *                if an input/output error occurs
     */
    public void encodeEnd(FacesContext context, UIComponent component)
            throws IOException {

        // Prepare the local variables we will need
        PopupCalendarComponent popCal = (PopupCalendarComponent) component;
        String clientId = popCal.getClientId(context);
        ResponseWriter writer = context.getResponseWriter();
        writer.startElement("div", popCal);
        writer.writeAttribute("id", clientId, "id");

        // Renders @style in particular
        renderStringPassThruAttributes(writer, popCal);

        // Render the CSS style class(es) (if any)
        String styleClass = popCal.getStyleClass();
        if (styleClass != null) {
            writer.writeAttribute("class", styleClass, "styleClass");
        }
        // TODO what about other attributes?

        // Render the required resource links once per page (runtime only)
        if (!Beans.isDesignTime()) {
            helper.linkStylesheet(context, component, writer,
                    Mechanism.CLASS_RESOURCE, CALENDAR_CSS);
            helper.linkStylesheet(context, component, writer,
                    Mechanism.CLASS_RESOURCE, POP_CAL_CSS);

            helper.linkJavascript(context, component, writer,
                    Mechanism.CLASS_RESOURCE, Util.UI_COMMON_SCRIPT_RESOURCE);
            Util.renderDojoLoading(context, component, writer);

            writer.startElement("script", component);
            writer.writeAttribute("type", "text/javascript", null);
            writer.write("\n");
            writer.write("dojo.setModulePrefix('" + POP_CAL_MODULE_NAME
                    + "', '../" + POP_CAL_MODULE_NAME + "');");
            writer.write("\n");

            // This tries to load in the module package file first which has the
            // dojo.widget.manager.registerWidgetPackage() call
            writer.write("dojo.require('" + POP_CAL_MODULE_NAME + ".*');");
            writer.write("\n");

            writer.endElement("script");
            writer.write("\n");
        }

        // Render an input text field
        writer.startElement("input", popCal);
        writer.writeAttribute("id", popCal.getInputTextClientId(context), null);
        writer.writeAttribute("type", "text", null);
        writer.writeAttribute("name", popCal.getInputTextClientId(context),
                null);
        writer.writeAttribute("value", popCal.getInputTextValue(), null);
        writer.writeAttribute("title", popCal.getInputTextTooltip(), null);
        writer.writeAttribute("class", POP_CAL_INPUT_TEXT_CLASS, null);
        writer.endElement("input");
        writer.write("\n");

        // Render a javascript image button
        writer.startElement("img", popCal);
        writer.writeAttribute("src", getImageUrl(context, component,
                CALENDAR_ICON), null);
        writer.writeAttribute("class", POP_CAL_BUTTON_CLASS, null);
        writer.writeAttribute("alt",
                Util.getMessage("popupcalendar.buttonAlt"), null);
        String script = popCal.getJavaScriptObjectName(context)
                + ".togglePopup();return false;";
        writer.writeAttribute("onclick", script, null);
        writer.endElement("img");
        writer.write("\n");

        // Render a div to use as a target location for the popup
        writer.startElement("div", popCal);
        writer.writeAttribute("id", popCal.getJavaScriptObjectName(context)
                + POP_CAL_CALENDAR_SUFFIX, null);
        writer.writeAttribute("class", POP_CAL_CALENDAR_CLASS, null);
        writer.endElement("div");
        writer.write("\n");

        renderScript(writer, component, popCal
                .getJavaScriptForNewInstance(context));

        // Render the end of top level div
        writer.endElement("div");
        writer.write("\n");
    }

    // ------------------------------------------------------==- Private Methods

    /**
     * <p>
     * Render pass-through HTML attributes with String values.
     * </p>
     * 
     * @param writer
     *            <code>ResponseWriter</code> to render with
     * @param component
     *            <code>UIComponent</code> being rendered
     */
    private void renderStringPassThruAttributes(ResponseWriter writer,
        UIComponent component) throws IOException {

        Object value;
        for (int i = 0; i < passthruAttributes.length; i++) {
            value = component.getAttributes().get(passthruAttributes[i]);
            if (value != null) {
                if (!(value instanceof String)) {
                    value = value.toString();
                }
                writer.writeAttribute(passthruAttributes[i], (String) value,
                        passthruAttributes[i]);
            }
        }

    }

    private void renderScript(ResponseWriter writer, UIComponent component,
        String script) throws IOException {
        writer.startElement("script", component);
        writer.writeAttribute("type", "text/javascript", null);
        writer.write("\n");
        writer.write(script);
        writer.endElement("script");
        writer.write("\n");
    }

    private String getImageUrl(FacesContext context, UIComponent component,
        String resourcePath) {
        if (Beans.isDesignTime()) {
            return component.getClass().getResource(resourcePath).toString();
        } else {
            return helper.mapResourceId(context, Mechanism.CLASS_RESOURCE,
                    resourcePath);
        }
    }
}
