/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: ProgressBarComponent.java,v 1.11 2006/05/02 11:01:04 mattbohm Exp $
 */

package com.sun.j2ee.blueprints.ui.progressbar;

import javax.faces.component.UICommand;
import javax.faces.context.FacesContext;
import javax.faces.el.ValueBinding;
import javax.faces.el.MethodBinding;

/**
 * <p>Component that shows the progress of a long-running process.</p>
 **/
public class ProgressBarComponent extends UICommand {
    
    /**
     * <p>The standard component family for this component.</p>
     */
    public static final String COMPONENT_FAMILY = "com.sun.j2ee.blueprints.ui.progressbar.ProgressBar";  //NOI18N
    
    /**
     * <p>The standard renderer type for this component.</p>
     */
    public static final String RENDERER_TYPE = "com.sun.j2ee.blueprints.ui.progressbar.ProgressBar";  //NOI18N
    
    /**
     * <p>The default indeterminate section value.</p>
     */
    public static final int INDETERMINATE_SECTION_DEFAULT = 10;
    
    /**
     * <p>The default indeterminate interval value.</p>
     */
    public static final int INDETERMINATE_INTERVAL_DEFAULT = 20;
    
    /**
     * <p>The default indeterminate increment value.</p>
     */
    public static final int INDETERMINATE_INCREMENT_DEFAULT = 1;
    
    // --------------------------------------------------------------Constructors
    
    public ProgressBarComponent() {
        super();
        setRendererType(RENDERER_TYPE);
    }
    
    
    // -------------------------------------------------------------- Properties
    
    /**
     * <p>Return the component family for this component.</p>
     */
    public String getFamily() {
        return (COMPONENT_FAMILY);
    }
    
    /**
     * <p>Return the <code>ValueBinding</code> stored for the
     * specified name (if any), respecting any property aliases.</p>
     *
     * @param name Name of value binding to retrieve
     */
    public ValueBinding getValueBinding(String name) {
        if (name.equals("percentage")) {  //NOI18N
            return super.getValueBinding("value");  //NOI18N
        }
        return super.getValueBinding(name);
    }
    
    /**
     * <p>Set the <code>ValueBinding</code> stored for the
     * specified name (if any), respecting any property
     * aliases.</p>
     *
     * @param name Name of value binding to set
     * @param binding ValueBinding to set, or null to remove
     */
    public void setValueBinding(String name,ValueBinding binding) {
        if (name.equals("percentage")) {  //NOI18N
            super.setValueBinding("value", binding);  //NOI18N
            return;
        }
        super.setValueBinding(name, binding);
    }
    
    
    //interval
    private int interval = 1000;
    
    /**
     * <p>Return the polling interval (in milliseconds) between ajax requests.</p>
     **/
    public int getInterval() {
        ValueBinding vb = getValueBinding("interval"); //NOI18N
        if (vb != null) {
            Object val = vb.getValue(getFacesContext());
            if (val instanceof Number) {
                return ((Number)val).intValue();
            }
        }
        return interval;
    }
    
    /**
     * <p>Set the polling interval (in milliseconds) between ajax requests.</p>
     **/
    public void setInterval(int newInterval) {
        interval = newInterval;
    }
    
    /**
     * <p>Return the percentage of completion to be shown in the progress bar.</p>
     **/
    public int getPercentage() {
        int percent = 67;   //default to 67 for designtime and runtime
        Object val = getValue();
        //System.err.println("ProgressBarComponent.getPercentage: value is " + val + ", and value class is " + (val == null ? "n/a" : String.valueOf(val.getClass())));
        if (val instanceof Number) {
            percent = ((Number)val).intValue();
        }
        return percent;
    }
    
    /**
     * <p>Set the percentage of completion to be shown in the progress bar.</p>
     **/
    public void setPercentage(int percentage) {
        setValue(new Integer(percentage));        
    }
    
    private String percentageText;
    
    /**
     * <p>Get text that indicates the progress made so far.
     * </p>
     */
    public String getPercentageText() {
        ValueBinding vb = getValueBinding("percentageText");  //NOI18N
        if (vb != null) {
            return (String) vb.getValue(getFacesContext());
        }
        return this.percentageText;
    }
    
    /**
     * <p>Set text that indicates the progress made so far.
     * </p>
     * 
     */
    public void setPercentageText(String percentageText) {
        this.percentageText = percentageText;
    }
    
    private MethodBinding startOperation;
    
    /**
     * <p>Get a method binding that is invoked to start
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public MethodBinding getStartOperation() {
        return this.startOperation;
    }
    
    /**
     * <p>Set a method binding that is invoked to start
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public void setStartOperation(MethodBinding startOperation) {
        this.startOperation = startOperation;
    }
    
    private MethodBinding pauseOperation;
    
    /**
     * <p>Get a method binding that is invoked to pause
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public MethodBinding getPauseOperation() {
        return this.pauseOperation;
    }
    
    /**
     * <p>Set a method binding that is invoked to pause
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public void setPauseOperation(MethodBinding pauseOperation) {
        this.pauseOperation = pauseOperation;
    }
    
    private MethodBinding resumeOperation;
    
    /**
     * <p>Get a method binding that is invoked to resume
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public MethodBinding getResumeOperation() {
        return this.resumeOperation;
    }
    
    /**
     * <p>Set a method binding that is invoked to resume
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public void setResumeOperation(MethodBinding resumeOperation) {
        this.resumeOperation = resumeOperation;
    }
    
    private MethodBinding stopOperation;
    
    /**
     * <p>Get a method binding that is invoked to stop
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public MethodBinding getStopOperation() {
        return this.stopOperation;
    }
    
    /**
     * <p>Set a method binding that is invoked to stop
     * the long-running process. The method signature must be:
     * <code>public void xxxx(FacesContext fc)</code>. You can obtain 
     * parameters, including the progress bar client ID 
     * (bpui_progressbar_clientId), from the external request parameter map.
     * </p>
     */
    public void setStopOperation(MethodBinding stopOperation) {
        this.stopOperation = stopOperation;
    }
    
    private String onComplete;
    
    /**
     * <p>Get a scripting function that will be executed when progress 
     * is complete.
     * </p>
     */
    public String getOnComplete() {
        ValueBinding vb = getValueBinding("onComplete");  //NOI18N
        if (vb != null) {
            return (String) vb.getValue(getFacesContext());
        }
        return this.onComplete;
    }
    
    /**
     * <p>Set a scripting function that will be executed when progress 
     * is complete.
     * </p>
     * 
     */
    public void setOnComplete(String onComplete) {
        this.onComplete = onComplete;
    }
    
    private String onFail;
    
    /**
     * <p>Get a scripting function that will be executed when progress 
     * fails.
     * </p>
     */
    public String getOnFail() {
        ValueBinding vb = getValueBinding("onFail");  //NOI18N
        if (vb != null) {
            return (String) vb.getValue(getFacesContext());
        }
        return this.onFail;
    }
    
    /**
     * <p>Set a scripting function that will be executed when progress 
     * fails.
     * </p>
     */
    public void setOnFail(String onFail) {
        this.onFail = onFail;
    }
    
    private String onUpdate;
    
    /**
     * <p>Get a scripting function that will be executed each time progress is
     * updated.
     * </p>
     */
    public String getOnUpdate() {
        ValueBinding vb = getValueBinding("onUpdate");  //NOI18N
        if (vb != null) {
            return (String) vb.getValue(getFacesContext());
        }
        return this.onUpdate;
    }
    
    /**
     * <p>Set a scripting function that will be executed each time progress 
     * is updated.
     * </p>
     * 
     */
    public void setOnUpdate(String onUpdate) {
        this.onUpdate = onUpdate;
    }
    
    private String onPoll;
    
    /**
     * <p>Get a scripting function that will be executed each time progress is
     * updated.
     * </p>
     */
    public String getOnPoll() {
        ValueBinding vb = getValueBinding("onPoll");  //NOI18N
        if (vb != null) {
            return (String) vb.getValue(getFacesContext());
        }
        return this.onPoll;
    }
    
    /**
     * <p>Set a scripting function that will be executed each time progress 
     * is updated.
     * </p>
     * 
     */
    public void setOnPoll(String onPoll) {
        this.onPoll = onPoll;
    }
    
    private boolean autoStart;
    
    /**
     * <p>Get whether the component should begin polling when the page 
     * loads.
     * </p>
     **/
    public boolean isAutoStart() {
        ValueBinding vb = getValueBinding("autoStart"); //NOI18N
        if (vb != null) {
            Boolean value = (Boolean)vb.getValue(getFacesContext());
            if (value != null) {
                return value.booleanValue();
            }
        }
        return this.autoStart;
    }
    
    /**
     * <p>Set whether the component should begin polling when the page 
     * loads.
     * </p>
     **/
    public void setAutoStart(boolean autoStart) {
        this.autoStart = autoStart;
    }
    
    private boolean failed;
    
    /**
     * <p>Get whether the operation has failed. This property, if used, 
     * must be bound.
     * </p>
     **/
    public boolean isFailed() {
        ValueBinding vb = getValueBinding("failed"); //NOI18N
        if (vb != null) {
            Boolean value = (Boolean)vb.getValue(getFacesContext());
            if (value != null) {
                return value.booleanValue();
            }
        }
        return this.failed;
    }
    
    /**
     * <p>Set whether the operation has failed. This property, if used, 
     * must be bound.
     * </p>
     **/
    public void setFailed(boolean failed) {
        this.failed = failed;
    }    
    
    private int indeterminateSection = INDETERMINATE_SECTION_DEFAULT;
    
    /**
     * <p>Get a percentage indicating the length of the bar's middle section 
     * for an indeterminate case.</p>
     **/
    public int getIndeterminateSection() {
        ValueBinding vb = getValueBinding("indeterminateSection"); //NOI18N
        if (vb != null) {
            Object val = vb.getValue(getFacesContext());
            if (val instanceof Number) {
                return ((Number)val).intValue();
            }
        }
        return indeterminateSection;
    }
    
    /**
     * <p>Set a percentage indicating the length of the bar's middle section 
     * for an indeterminate case.</p>
     **/
    public void setIndeterminateSection(int indeterminateSection) {
        this.indeterminateSection = indeterminateSection;
    }
    
    private int indeterminateInterval = INDETERMINATE_INTERVAL_DEFAULT;
    
    /**
     * <p>Get the interval in milliseconds between updates to the bar's 
     * display for an indeterminate case.</p>
     **/
    public int getIndeterminateInterval() {
        ValueBinding vb = getValueBinding("indeterminateInterval"); //NOI18N
        if (vb != null) {
            Object val = vb.getValue(getFacesContext());
            if (val instanceof Number) {
                return ((Number)val).intValue();
            }
        }
        return indeterminateInterval;
    }
    
    /**
     * <p>Set the interval in milliseconds between updates to the bar's 
     * display for an indeterminate case.</p>
     **/
    public void setIndeterminateInterval(int indeterminateInterval) {
        this.indeterminateInterval = indeterminateInterval;
    }
    
    private int indeterminateIncrement = INDETERMINATE_INCREMENT_DEFAULT;
    
    /**
     * <p>Get the percentage by which the bar's display will be incremented 
     * for an indeterminate case.</p>
     **/
    public int getIndeterminateIncrement() {
        ValueBinding vb = getValueBinding("indeterminateIncrement"); //NOI18N
        if (vb != null) {
            Object val = vb.getValue(getFacesContext());
            if (val instanceof Number) {
                return ((Number)val).intValue();
            }
        }
        return indeterminateIncrement;
    }
    
    /**
     * <p>Set the percentage by which the bar's display will be incremented 
     * for an indeterminate case.</p>
     **/
    public void setIndeterminateIncrement(int indeterminateIncrement) {
        this.indeterminateIncrement = indeterminateIncrement;
    }
    
    // style
    private String style = null;   //NOI18N
    
    /**
     * <p>CSS style(s) to be applied when this component is rendered.</p>
     */
    public String getStyle() {
        if (this.style != null) {
            return this.style;
        }
        ValueBinding vb = getValueBinding("style"); //NOI18N
        if (vb != null) {
            return (String) vb.getValue(getFacesContext());
        }
        return null;
    }
    
    /**
     * <p>CSS style(s) to be applied when this component is rendered.</p>
     * @see #getStyle()
     */
    public void setStyle(String style) {
        this.style = style;
    }
    
    // styleClass
    private String styleClass = null;
    
    /**
     * <p>CSS style class(es) to be applied when this component is rendered.</p>
     */
    public String getStyleClass() {
        if (this.styleClass != null) {
            return this.styleClass;
        }
        ValueBinding vb = getValueBinding("styleClass");  //NOI18N
        if (vb != null) {
            return (String) vb.getValue(getFacesContext());
        }
        return null;
    }
    
    /**
     * <p>CSS style class(es) to be applied when this component is rendered.</p>
     * @see #getStyleClass()
     */
    public void setStyleClass(String styleClass) {
        this.styleClass = styleClass;
    }
    
    // ----------------------------------------------------- StateHolder Methods
    /**
     * <p>Return the state to be saved for this component.
     *
     * @param context <code>FacesContext</code> for the current request
     */
    public Object saveState(FacesContext context) {
        Object values[] = new Object[18];
        values[0] = super.saveState(context);
        values[1] = new Integer(interval);
        values[2] = this.percentageText;
        values[3] = saveAttachedState(context, this.startOperation);
        values[4] = saveAttachedState(context, this.pauseOperation);
        values[5] = saveAttachedState(context, this.resumeOperation);
        values[6] = saveAttachedState(context, this.stopOperation);
        values[7] = this.onComplete;
        values[8] = this.onFail;
        values[9] = this.onUpdate;
        values[10] = this.onPoll;
        values[11] = new Boolean(this.autoStart);
        values[12] = new Boolean(this.failed);
        values[13] = new Integer(indeterminateSection);
        values[14] = new Integer(indeterminateInterval);
        values[15] = new Integer(indeterminateIncrement);
        values[16] = this.style;
        values[17] = this.styleClass;
        return (values);
    }
    
    public void restoreState(FacesContext context, Object state) {
        Object values[] = (Object[]) state;
        super.restoreState(context, values[0]);
        this.interval = ((Integer)values[1]).intValue();
        this.percentageText = (String)values[2];
        this.startOperation = (MethodBinding) restoreAttachedState(context, values[3]);
        this.pauseOperation = (MethodBinding) restoreAttachedState(context, values[4]);
        this.resumeOperation = (MethodBinding) restoreAttachedState(context, values[5]);
        this.stopOperation = (MethodBinding) restoreAttachedState(context, values[6]);
        this.onComplete = (String)values[7];
        this.onFail = (String)values[8];
        this.onUpdate = (String)values[9];
        this.onPoll = (String)values[10];
        this.autoStart = ((Boolean)values[11]).booleanValue();
        this.failed = ((Boolean)values[12]).booleanValue();
        this.indeterminateSection = ((Integer)values[13]).intValue();
        this.indeterminateInterval = ((Integer)values[14]).intValue();
        this.indeterminateIncrement = ((Integer)values[15]).intValue();
        this.style = (String)values[16];
        this.styleClass = (String)values[17];
    }
}
