/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: SelectValueRenderer.java,v 1.12 2006/04/26 04:03:03 craig_mcc Exp $
 */

package com.sun.j2ee.blueprints.ui.selectvalue;

import java.beans.Beans;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.faces.component.UIInput;
import javax.faces.component.UIComponent;
import javax.faces.component.UIViewRoot;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.render.Renderer;
import javax.faces.model.SelectItem;
import javax.faces.FacesException;
import javax.faces.webapp.UIComponentTag;
import javax.faces.el.MethodBinding;
import com.sun.j2ee.blueprints.ui.util.BaseRenderer;
import com.sun.j2ee.blueprints.ui.util.Util;

import org.apache.shale.remoting.Mechanism;
import org.apache.shale.remoting.XhtmlHelper;

/**
 * This renderer generates the necessary markup (including JavaScript) 
 * for SelectValueComponent.
 * @author Jayashri Visvanathan
 */
public class SelectValueRenderer extends BaseRenderer {

    private static final String SELECT_SUFFIX = "_selectdiv";
    private static final String INPUT_SUFFIX = "_text";
    
    /**
     * <p>The static script resource for this component.</p>
     */
    private static final String SCRIPT_RESOURCE =
            "/META-INF/selectvalue/script.js";
    
    /**
     * <p>Resource identifier for our completion results event handler.</p>
     */
    private static final String SELECTVALUE_HANDLER =
            "/bpui_selectvalue_handler/matchValue";
    
    /**
     * <p>Stateless helper bean to manufacture resource linkages.</p>
     */
    private static XhtmlHelper helper = new XhtmlHelper();
    
    public SelectValueRenderer() {
    }

    public void decode(FacesContext context, UIComponent component) {

        if (context == null || component == null) {
            throw new NullPointerException();
        }
       
        if (!(component instanceof UIInput)) {
            // decode needs to be invoked only for components that are
            // instances or subclasses of UIInput.
            return;
        }    

        // If the component is disabled, do not change the value of the
        // component, since its state cannot be changed.
        if (componentIsDisabledOnReadonly(component)) {
            return;
        }

        String clientId = replaceColon(component.getClientId(context));
        clientId = clientId + INPUT_SUFFIX;
    
        Map requestMap = context.getExternalContext().getRequestParameterMap();
        // Don't overwrite the value unless you have to!
        if (requestMap.containsKey(clientId)) {
            String newValue = (String) requestMap.get(clientId);
            ((UIInput) component).setSubmittedValue(newValue);
        }
    }
    
    public void encodeBegin(FacesContext context, UIComponent component)
        throws IOException {
    }

    public void encodeChildren(FacesContext context, UIComponent component) {
    }

     public void encodeEnd(FacesContext context, UIComponent component)
        throws IOException {

        ResponseWriter writer = context.getResponseWriter();
        
        SelectValueComponent comp = (SelectValueComponent)component;
        String itemsMethod = null;
        if (comp.getItemsMethod() != null) {
            itemsMethod = comp.getItemsMethod().getExpressionString();
        }
        
        // don't render any JavaScript during design time.
        if (!Beans.isDesignTime()) {
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  Util.UI_COMMON_SCRIPT_RESOURCE);
            Util.renderDojoLoading(context, component, writer);
            helper.linkJavascript(context, component, writer,
                                  Mechanism.CLASS_RESOURCE,
                                  SCRIPT_RESOURCE);
            
            // Localize the error messages for this component
            writer.startElement("script", comp);
            writer.writeAttribute("type", "text/javascript", null);
            writer.write("\n");
            writer.write("bpui.selectvalue.messages[\"bindError\"] = \"" +
                         Util.getMessage("selectvalue.bindError") + "\";");
            writer.write("\n");
            writer.endElement("script");
            writer.write("\n");
        }
        
        // Start an outer div for the entire comopnent
        writer.startElement("div", component);
        
        // write the "id" of the SelectValue component as the "id" of the
        // outer "div" element.
        String clientId = replaceColon(component.getClientId(context));
        writer.writeAttribute("id", clientId, "id");
     
        // write the styleclass and style of the SelectValue component on
        // outer "div" element.
        String styleClass = (String)component.getAttributes().get("styleClass");
        if (null != styleClass) {
            writer.writeAttribute("class", styleClass, "styleClass");
        }
        
        String width = null; 
        String height = null;
        String style = comp.getStyle();
        if (style != null) {
            writer.writeAttribute("style", style, "style");
            // get width and height by parsing style
            int beginIdx = style.indexOf("width");
            int endIdx = style.indexOf(";", beginIdx);
            // width could be the last attribute, if so, there won't exist a ";"
            if (endIdx == -1) {
                endIdx =style.length();
            }
            if (beginIdx != -1 && endIdx != 1) {
                width = style.substring(beginIdx, endIdx);
            }
            
            // get  height
            beginIdx = style.indexOf("height");
            endIdx = style.indexOf(";", beginIdx + 1);
            if (endIdx == -1) {
                endIdx =style.length();
            }
            if (beginIdx != -1 && endIdx != 1) {
                height = style.substring(beginIdx, endIdx);
            }
        }
        
        String id =  clientId + INPUT_SUFFIX;
        writer.startElement("input", component);
        // use clientId plus suffix as the id for the input.
        writer.writeAttribute("id", id, "id");
        String inputStyle = "";
        if (width != null) {
            inputStyle = inputStyle + width;
        }
        if ( height != null) {
            if (width != null) {
                inputStyle = inputStyle + ";";
            }
            inputStyle = inputStyle + height;
        }
       
        if (inputStyle != "") {
            writer.writeAttribute("style", inputStyle, "style");
        }
       
        // Set the autocomplete attribute to "off" to disable browser 
        // textfield completion with previous entries
        writer.writeAttribute("autocomplete", "off", null);

        writer.writeAttribute("type", "text", null);
        writer.writeAttribute("name", id, "clientId");

        Object text = comp.getValue();
        if (text != null) {
            writer.writeAttribute("value", text, "value");
        }

        // Emit the javascript for asynchronous method invocation upon "onchange"
        // and "onfocus" only if itemsMethod is not null
        if (!Beans.isDesignTime() && itemsMethod != null && 
                itemsMethod.length() > 0) { 
            String callback = helper.mapResourceId(context,
                                                   Mechanism.DYNAMIC_RESOURCE,
                                                   SELECTVALUE_HANDLER);
            String jsScript =
                "bpui.selectvalue.executeQuery('" + clientId + "','" 
                + itemsMethod + "'" + ",'" + callback + "');";
            
            // append any user specified script for "onchange"
            String onkeyup = (String)component.getAttributes().get("onkeyup");
            if (onkeyup != null && onkeyup.length() > 0) {
                onkeyup =jsScript + onkeyup;
            } else {
                onkeyup =jsScript;
            }
            writer.writeAttribute("onkeyup", onkeyup, null);
            
            // as soon as the element gets focus, process any initial value
            // set on the component.
            String onfocus = (String) component.getAttributes().get("onfocus"); 
            // PENDING (jayashri) should the method be invoked if the value
            // is not null null ?
           // jsScript = "bpui_selectvalue_setup_" + clientId + "();";
            if (onfocus != null && onfocus.length() > 0) {
                onfocus = jsScript + onfocus;
            } else {
                onfocus =jsScript;
            }
            writer.writeAttribute("onfocus", onfocus, null);
        }
        
        renderPassThruAttributes(writer, component, 
                new String[]{"onkeyup, onfocus"});
        renderBooleanPassThruAttributes(writer, component);
        writer.endElement("input");

        // render a div tag that will display the results of the method 
        // invocation during run time in a dropdown list.
        writer.startElement("div", component);
        writer.writeAttribute("id", (clientId + SELECT_SUFFIX), null);
        
        // render style class if  present.
        if (null != styleClass) {
            writer.writeAttribute("class", styleClass, "styleClass");
        }
        
        if (Beans.isDesignTime()) {
            // render a mock "select" tag for design time for better visualization.
             String item1 = Util.getMessage("selectvalue.design.selectitem1");
             String item2 = Util.getMessage("selectvalue.design.selectitem2");
             String item3 = Util.getMessage("selectvalue.design.selectitem3");
             writer.write("<br/><select id= '" + (clientId + "_select") + 
                     "' size=\"1\" >" +
                     "<option value='" + item1 + "'>" + item1 + "</option>" +
                     "<option value='" + item2 + "'>" + item2 + "</option>" +
                     "<option value='" + item3 + "'>" + item3 + "</option>" +
                     "</select>\n");
        } else {
            writer.writeAttribute("style", "display:none", null);
        }
        
        writer.endElement("div");
        writer.write("\n");
        
        // close outer "div" tag
        writer.endElement("div");
        writer.write("\n");     
    }
    
   /**
    * Replace any ":" with "_" as it ":" is not allowed in JavaScript.
    */
    private String replaceColon(String clientId) {

        return clientId.replace(':', '_');

    }

}
