/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: AbstractUIComponentTag.java,v 1.2 2006/01/19 20:01:45 craig_mcc Exp $
 */

package com.sun.j2ee.blueprints.ui.util;

import javax.faces.component.UIComponent;
import javax.faces.el.ValueBinding;
import javax.faces.webapp.UIComponentTag;

/**
 * <p>Convenient base class for JSP tag handlers that extend
 * <code>javax.faces.webapp.UIComponentTag</code>.  This class
 * handles the tag attributes common to all component tags, and
 * provides convenience methods for setting component properties
 * with either literal values or value binding expressions.</p>
 */
public abstract class AbstractUIComponentTag extends UIComponentTag {
    

    // ---------------------------------------------------------- Tag Attributes


    /**
     * <p>The CSS style(s) used to render this component.</p>
     */
    private String style = null;

    public void setStyle(String style) {
        this.style = style;
    }


    /**
     * <p>The CSS style class(es) used to render this component.</p>
     */
    private String styleClass = null;

    public void setStyleClass(String styleClass) {
        this.styleClass = styleClass;
    }


    // -------------------------------------------------- UIComponentTag Methods


    /**
     * <p>Return the component type for this tag.</p>
     */
    public abstract String getComponentType();


    /**
     * <p>Return the renderer type for this tag.</p>
     */
    public abstract String getRendererType();

    public String getStyle() {
        return style;
    }


    /**
     * <p>Release any variables we allocated.</p>
     */
    public void release() {

        super.release();
        this.style = null;
        this.styleClass = null;

    }


    /**
     * <p>Configure attributes set on this tag into the component instance.</p>
     *
     * @param component <code>UIComponent</code> being configured
     */
    protected void setProperties(UIComponent component) {

        super.setProperties(component);
        configureString(component, "style", style);
        configureString(component, "styleClass", styleClass);

    }


    // ------------------------------------------------------- Protected Methods


    /**
     * <p>Configure a component property of type <code>boolean</code>.</p>
     *
     * @param component <code>UIComponent</code> being configured
     * @param name Name of the property being configured
     * @param value Value or expression to configure
     */
    protected void configureBoolean(UIComponent component, String name, String value) {

        if (value == null) {
            return;
        }
        if (isValueReference(value)) {
            ValueBinding vb =
              getFacesContext().getApplication().createValueBinding(value);
            component.setValueBinding(name, vb);
        } else {
            component.getAttributes().put(name, Boolean.valueOf(value));
        }

    }


    /**
     * <p>Configure a component property of type <code>int</code>.</p>
     *
     * @param component <code>UIComponent</code> being configured
     * @param name Name of the property being configured
     * @param value Value or expression to configure
     */
    protected void configureInteger(UIComponent component, String name, String value) {

        if (value == null) {
            return;
        }
        if (isValueReference(value)) {
            ValueBinding vb =
              getFacesContext().getApplication().createValueBinding(value);
            component.setValueBinding(name, vb);
        } else {
            component.getAttributes().put(name, Integer.valueOf(value));
        }

    }


    /**
     * <p>Configure a component property of type <code>Object</code>.</p>
     *
     * @param component <code>UIComponent</code> being configured
     * @param name Name of the property being configured
     * @param value Value or expression to configure
     */
    protected void configureObject(UIComponent component, String name, String value) {

        if (value == null) {
            return;
        }
        if (isValueReference(value)) {
            ValueBinding vb =
              getFacesContext().getApplication().createValueBinding(value);
            component.setValueBinding(name, vb);
        } else {
            component.getAttributes().put(name, value);
        }

    }


    /**
     * <p>Configure a component property of type <code>String</code>.</p>
     *
     * @param component <code>UIComponent</code> being configured
     * @param name Name of the property being configured
     * @param value Value or expression to configure
     */
    protected void configureString(UIComponent component, String name, String value) {

        if (value == null) {
            return;
        }
        if (isValueReference(value)) {
            ValueBinding vb =
              getFacesContext().getApplication().createValueBinding(value);
            component.setValueBinding(name, vb);
        } else {
            component.getAttributes().put(name, value);
        }

    }


}
