/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * You may not modify, use, reproduce, or distribute this
 * software except in compliance with the terms of the License at:
 *
 *   http://developer.sun.com/berkeley_license.html
 *
 * $Id: Util.java,v 1.10 2006/04/26 04:03:04 craig_mcc Exp $
 */

package com.sun.j2ee.blueprints.ui.util;

import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.logging.Logger;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;
import javax.faces.el.ValueBinding;
import javax.faces.application.Application;
import javax.faces.el.MethodBinding;
import org.apache.shale.remoting.Mechanism;
import org.apache.shale.remoting.XhtmlHelper;

/**
 *
 * @author edburns
 */
public class Util {


    public static final String UI_MESSAGE_STRINGS =
            "com.sun.j2ee.blueprints.ui.MessageStrings";

    public static final String UTIL_LOGGER = "com.sun.j2ee.blueprints";

    public static final String UI_LOG_STRINGS =
            "com.sun.j2ee.blueprints.ui.LogStrings";

    public static final String UI_LOGGER = "com.sun.j2ee.blueprints.ui.";

    public static final String UI_COMMON_DOJO_RESOURCE =
            "/META-INF/dojo/dojo.js";

    public static final String UI_COMMON_SCRIPT_RESOURCE =
            "/META-INF/common/script.js";

    // Log instance for this class
    private static Logger logger = getLogger(UTIL_LOGGER);


    public static Logger getLogger( String loggerName ) {
        return Logger.getLogger(loggerName, UI_LOG_STRINGS );
    }

    private static XhtmlHelper helper = new XhtmlHelper();
    
    /**
     * This array contains attributes that have a boolean value in JSP,
     * but have have no value in HTML.  For example "disabled" or
     * "readonly". <P>
     *
     * @see renderBooleanPassthruAttributes
     */

    private static String booleanPassthruAttributes[] = {
        "disabled",
        "readonly",
        "ismap"
    };

    /**
     * This array contains attributes whose value is just rendered
     * straight to the content.  This array should only contain
     * attributes that require no interpretation by the Renderer.  If an
     * attribute requires interpretation by a Renderer, it should be
     * removed from this array.<P>
     *
     * @see renderPassthruAttributes
     */
    private static String passthruAttributes[] = {
        "accept", "accesskey","alt", "bgcolor","border", "cellpadding", 
        "cellspacing","charset", "cols", "coords", "dir", "enctype", "frame", 
        "height", "hreflang", "lang", "xml","longdesc", "maxlength", "onblur", 
        "onchange", "onclick", "ondblclick", "onfocus", "onkeydown","onkeypress", 
        "onkeyup", "onload", "onmousedown", "onmousemove", "onmouseout", 
        "onmouseover", "onmouseup", "onreset", "onselect", "onsubmit", 
        "onunload", "rel", "rev", "rows", "rules", "shape", "size", "style", 
        "summary", "tabindex", "target", "title", "usemap", "width"
    };

//
// Constructors and Initializers    
//

    private Util() {
        throw new IllegalStateException();
    }

    /**
     * <p>Render any boolean "passthru" attributes.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param component <code>UIComponent</code> being rendered
     *
     */

    public static String renderBooleanPassthruAttributes(FacesContext context,
                                                         UIComponent component) {
        int i = 0, len = booleanPassthruAttributes.length;
        String value;
        boolean thisIsTheFirstAppend = true;
        StringBuffer renderedText = new StringBuffer();

        for (i = 0; i < len; i++) {
            if (null != (value = (String)
                component.getAttributes().get(booleanPassthruAttributes[i]))) {
                if (thisIsTheFirstAppend) {
                    // prepend ' '
                    renderedText.append(' ');
                    thisIsTheFirstAppend = false;
                }
                if (Boolean.valueOf(value).booleanValue()) {
                    renderedText.append(booleanPassthruAttributes[i] + ' ');
                }
            }
        }

        return renderedText.toString();
    }


    /**
     * <p>Render script to dynamically load DOJO if it has not already
     * been loaded (which could have been by the user manually including
     * it, so we cannot rely on a request scope attribute to check).</p>
     *
     * <p><strong>WARNING</strong> - The JavaScript code rendered by this
     * method assumes you have previously loaded the UI_COMMON_SCRIPT_RESOURCE
     * JavaScript resource.</p>
     *
     * @param context <code>FacesContext</code> for the current request
     * @param cmponent <code>UIComponent</code> to associate with this
     *  rendered script
     * @param writer <code>ResponseWriter for the current page
     *
     * @exception IOException if an input/output error occurs
     */
    public static void renderDojoLoading(FacesContext context,
            UIComponent component, ResponseWriter writer) throws IOException {

        writer.startElement("script", component);
        writer.writeAttribute("type", "text/javascript", null);
        writer.write("\n");

        writer.write("if (typeof dojo == 'undefined') {\n");

//        writer.write("    document.write(\"<script type='text/javascript' src='" +
//          helper.mapResourceId(context, Mechanism.CLASS_RESOURCE, UI_COMMON_DOJO_RESOURCE) +
//          "'></script>\");\n");

        writer.write("    bpui.load(\"" +
          helper.mapResourceId(context, Mechanism.CLASS_RESOURCE, UI_COMMON_DOJO_RESOURCE) +
          "\");\n");

        writer.write("}\n");

        writer.endElement("script");
        writer.write("\n");

    }


    /**
     * Render any "passthru" attributes, where we simply just output the
     * raw name and value of the attribute.  This method is aware of the
     * set of HTML4 attributes that fall into this bucket.  Examples are
     * all the javascript attributes, alt, rows, cols, etc.  <P>
     *
     * @return the rendererd attributes as specified in the component.
     *         Padded with leading and trailing ' '.  If there are no passthru
     *         attributes in the component, return the empty String.
     */

    public static String renderPassthruAttributes(FacesContext context,
                                                  UIComponent component) {
        int i = 0, len = passthruAttributes.length;
        String value;
        boolean thisIsTheFirstAppend = true;
        StringBuffer renderedText = new StringBuffer();

        for (i = 0; i < len; i++) {
            if (null != (value = (String)
                component.getAttributes().get(passthruAttributes[i]))) {
                if (thisIsTheFirstAppend) {
                    // prepend ' '
                    renderedText.append(' ');
                    thisIsTheFirstAppend = false;
                }
                renderedText.append(passthruAttributes[i] + "=\"" + value +
                                    "\" ");
            }
        }

        return renderedText.toString();
    }

    public static ValueBinding getValueExpression(String exprString,
                                                     Class expectedType,
                                                     FacesContext context) {
            ValueBinding ve =
                  context.getApplication().createValueBinding(exprString);
        return ve;
    }

    
    private static final String RENDERED_RESOURCE = "com.sun.j2ee.blueprints.RENDERED_RESOURCE";        

    /** <p>Render a resource once per request, such as a javascript &lt;script&gt; tag.</p>
    * @param context The FacesContext
    * @param writer The ResponseWriter to use
    * @param component The component that needs the resource (used for designtime purposes).
    * @param pathToResource The path to the resource, e.g., "/META-INF/autocomplete/script.js"
    * @param contentType The content type, e.g., "text/javascript"
    */
    public static void renderResourceOnce(FacesContext context, 
            ResponseWriter writer,
            UIComponent component,  
            String pathToResource,
            String contentType) throws IOException {

        String key = RENDERED_RESOURCE + pathToResource;
        if (!context.getExternalContext().getRequestMap().containsKey(key)) {

            context.getExternalContext().getRequestMap().put(key, Boolean.TRUE);

            if ("text/css".equals(contentType)) {
                writer.startElement("link", component);
                writer.writeAttribute("type", contentType, null);
                writer.writeAttribute("rel", "stylesheet", null);
                StringBuffer src = new StringBuffer();
                src.append(context.getExternalContext().getRequestContextPath());
                src.append("/faces/resource?r=");
                src.append(pathToResource);
                writer.writeAttribute("href", src.toString(), null);
                writer.endElement("link");
                writer.write("\n");
            }
            else {
                writer.startElement("script", component);
                writer.writeAttribute("type", contentType, null);
                StringBuffer src = new StringBuffer();
                src.append(context.getExternalContext().getRequestContextPath());
                src.append("/faces/resource?r=");
                src.append(pathToResource);
                writer.writeAttribute("src", src.toString(), null);
                writer.endElement("script");
                writer.write("\n");
            }

        }

    }

    public static MethodBinding createConstantMethodBinding(String outcome) {
        return new ConstantMethodBinding(outcome);
    }


    /**
     * <p>Cache of previously looked up resource bundles, keyed by
     * <code>Locale</code>.</p>
     */
    private static Map bundles = new HashMap();


    /**
     * <p>Return the message text for the specified key, localized by the
     * <code>Locale</code> in the view root of the current response.</p>
     *
     * @param key Message key to return
     */
    public static String getMessage(String key) {

        Locale locale = FacesContext.getCurrentInstance().getViewRoot().getLocale();
        if (locale == null) {
            locale = Locale.getDefault();
        }
        ResourceBundle bundle = (ResourceBundle) bundles.get(locale);
        if (bundle == null) {
            ClassLoader cl = Util.class.getClassLoader();
            bundle = ResourceBundle.getBundle(UI_MESSAGE_STRINGS, locale, cl);
            bundles.put(locale, bundle);
        }
        return bundle.getString(key);

    }


}
